'use client';
import { useState, useMemo } from 'react';
import { useInventory } from '../providers/inventory-provider';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  Tooltip,
  Legend,
  LineChart,
  Line,
  PieChart,
  Pie,
  Cell,
  Label,
  ScatterChart,
  Scatter,
  ZAxis,
  Brush,
} from 'recharts';
import { ChartContainer, ChartTooltipContent } from '../ui/chart';
import { aggregateInventoryData } from '@/lib/inventory-calculations';
import type { ExtendedInventoryData } from '@/lib/inventory-calculations';
import { ReorderProfitabilityChart } from './reorder-profitability-chart';

type ChartType = 'bar' | 'line' | 'pie' | 'scatter' | 'reorder';
type Metric =
  | 'Revenue'
  | 'Profit'
  | 'Sales (Units)'
  | 'Closing Stock (Units)'
  | 'Opening Stock (Units)'
  | 'Purchase (Units)'
  | 'Cost of Goods Sold'
  | 'Profit Margin (%)'
  | 'Stock Turnover Ratio'
  | 'Reorder Level';

type Dimension = 'Item Name' | 'SKU' | 'Category';

const COLORS = [
  'hsl(var(--chart-1))',
  'hsl(var(--chart-2))',
  'hsl(var(--chart-3))',
  'hsl(var(--chart-4))',
  'hsl(var(--chart-5))',
];

const CURRENCY_METRICS: Metric[] = ['Revenue', 'Profit', 'Cost of Goods Sold'];

export function CustomChartsView() {
  const { inventoryData } = useInventory();
  const [chartType, setChartType] = useState<ChartType>('bar');
  const [metric, setMetric] = useState<Metric>('Revenue');
  const [dimension, setDimension] = useState<Dimension>('Item Name');

  // Additional state for scatter plot
  const [scatterXMetric, setScatterXMetric] =
    useState<Metric>('Sales (Units)');
  const [scatterYMetric, setScatterYMetric] = useState<Metric>('Profit');

  const aggregatedData = useMemo(() => {
    if (chartType === 'scatter' || chartType === 'reorder') {
      return aggregateInventoryData(inventoryData, dimension, metric, true);
    }
    return aggregateInventoryData(inventoryData, dimension, metric, false);
  }, [inventoryData, dimension, metric, chartType]);

  const scatterPlotData = useMemo(() => {
    return aggregateInventoryData(
      inventoryData,
      dimension,
      'Revenue', // a dummy metric, as scatter will use its own
      true
    ) as ExtendedInventoryData[];
  }, [inventoryData, dimension]);

  const formatCurrency = (value: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(value);
  };
  
  const formatNumber = (value: number) => {
    return value.toLocaleString('en-IN');
  };

  const getAxisFormatter = (metricName: Metric) => {
    return CURRENCY_METRICS.includes(metricName) ? formatCurrency : formatNumber;
  }
  
  const getTooltipFormatter = (metricName: Metric) => {
    const isCurrency = CURRENCY_METRICS.includes(metricName)
    return (value: any) => {
       if (typeof value === 'number') {
        return isCurrency ? formatCurrency(value) : formatNumber(value);
       }
       return value.toString();
    }
  }


  const renderChart = () => {
    switch (chartType) {
      case 'bar':
        return (
          <ChartContainer
            config={{ [metric]: { label: metric, color: 'hsl(var(--chart-1))' } }}
            className="min-h-[400px] w-full"
          >
            <BarChart
              data={aggregatedData}
              margin={{ top: 20, right: 20, bottom: 60, left: 20 }}
              animationDuration={500}
              animationEasing="ease-in-out"
            >
              <XAxis
                dataKey="name"
                angle={-45}
                textAnchor="end"
                interval={0}
                height={100}
              />
              <YAxis tickFormatter={getAxisFormatter(metric)} />
              <Tooltip
                content={
                  <ChartTooltipContent
                    formatter={getTooltipFormatter(metric)}
                  />
                }
              />
              <Legend />
              <Brush
                dataKey="name"
                height={30}
                stroke="hsl(var(--primary))"
                startIndex={0}
                endIndex={Math.min(15, aggregatedData.length - 1)}
              />
              <Bar
                dataKey="value"
                name={metric}
                fill="hsl(var(--chart-1))"
                radius={4}
              />
            </BarChart>
          </ChartContainer>
        );
      case 'line':
        return (
          <ChartContainer
            config={{ [metric]: { label: metric, color: 'hsl(var(--chart-2))' } }}
            className="min-h-[400px] w-full"
          >
            <LineChart
              data={aggregatedData}
              margin={{ top: 20, right: 20, bottom: 60, left: 20 }}
              animationDuration={500}
              animationEasing="ease-in-out"
            >
              <XAxis
                dataKey="name"
                angle={-45}
                textAnchor="end"
                interval={0}
                height={100}
              />
              <YAxis tickFormatter={getAxisFormatter(metric)} />
              <Tooltip
                 content={
                  <ChartTooltipContent
                    formatter={getTooltipFormatter(metric)}
                  />
                }
              />
              <Legend />
              <Brush
                dataKey="name"
                height={30}
                stroke="hsl(var(--primary))"
                startIndex={0}
                endIndex={Math.min(15, aggregatedData.length - 1)}
              />
              <Line
                type="monotone"
                dataKey="value"
                name={metric}
                stroke="hsl(var(--chart-2))"
              />
            </LineChart>
          </ChartContainer>
        );
      case 'pie':
        const totalValue = (
          aggregatedData as { name: string; value: number }[]
        ).reduce((sum, entry) => sum + entry.value, 0);
        return (
          <ChartContainer
            config={{}}
            className="min-h-[400px] w-full flex items-center justify-center"
          >
            <PieChart width={400} height={400} animationDuration={500} animationEasing="ease-in-out">
              <Pie
                data={aggregatedData as { name: string; value: number }[]}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }) =>
                  `${name}: ${(percent * 100).toFixed(0)}%`
                }
                outerRadius={150}
                dataKey="value"
                nameKey="name"
              >
                {(aggregatedData as { name: string; value: number }[]).map(
                  (entry, index) => (
                    <Cell
                      key={`cell-${index}`}
                      fill={COLORS[index % COLORS.length]}
                    />
                  )
                )}
                <Label
                  content={({ viewBox }) => {
                    if (viewBox && 'cx' in viewBox && 'cy' in viewBox) {
                      return (
                        <text
                          x={viewBox.cx}
                          y={viewBox.cy}
                          textAnchor="middle"
                          dominantBaseline="middle"
                        >
                          <tspan
                            x={viewBox.cx}
                            dy="-0.5em"
                            className="text-2xl font-bold fill-foreground"
                          >
                            {getAxisFormatter(metric)(totalValue)}
                          </tspan>
                          <tspan
                            x={viewBox.cx}
                            dy="1.2em"
                            className="text-sm fill-muted-foreground"
                          >
                            Total {metric}
                          </tspan>
                        </text>
                      );
                    }
                  }}
                />
              </Pie>
              <Tooltip
                content={
                  <ChartTooltipContent
                     formatter={getTooltipFormatter(metric)}
                  />
                }
              />
              <Legend />
            </PieChart>
          </ChartContainer>
        );
      case 'scatter':
        return (
          <ChartContainer
            config={{}}
            className="min-h-[450px] w-full"
          >
            <ScatterChart 
              margin={{ top: 20, right: 40, bottom: 60, left: 20 }}
              animationDuration={500}
              animationEasing="ease-in-out"
            >
              <XAxis
                type="number"
                dataKey={scatterXMetric}
                name={scatterXMetric}
                tickFormatter={getAxisFormatter(scatterXMetric)}
                angle={-45} textAnchor="end" height={100}
              />
              <YAxis
                type="number"
                dataKey={scatterYMetric}
                name={scatterYMetric}
                tickFormatter={getAxisFormatter(scatterYMetric)}
              />
              <ZAxis type="category" dataKey={dimension} name={dimension} />
              <Tooltip
                cursor={{ strokeDasharray: '3 3' }}
                content={<ChartTooltipContent />}
              />
              <Legend />
              <Scatter
                name="Products"
                data={scatterPlotData}
                fill="hsl(var(--chart-1))"
              />
            </ScatterChart>
          </ChartContainer>
        );
        case 'reorder':
          return <ReorderProfitabilityChart />;
      default:
        return null;
    }
  };

  return (
    <div className="grid auto-rows-max items-start gap-4 md:gap-8 lg:col-span-2">
      <Card className="animate-in fade-in-0 slide-in-from-top-4 duration-500">
        <CardHeader>
          <CardTitle>Custom Chart Builder</CardTitle>
          <CardDescription>
            Create your own charts to analyze inventory data. Use the brush to zoom into bar and line charts.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid md:grid-cols-3 gap-4 mb-8">
            <div className="flex flex-col gap-2">
              <label htmlFor="chart-type" className="text-sm font-medium">
                Chart Type
              </label>
              <Select
                value={chartType}
                onValueChange={(v) => setChartType(v as ChartType)}
              >
                <SelectTrigger id="chart-type">
                  <SelectValue placeholder="Select chart type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="bar">Bar Chart</SelectItem>
                  <SelectItem value="line">Line Chart</SelectItem>
                  <SelectItem value="pie">Pie Chart</SelectItem>
                  <SelectItem value="scatter">Scatter Plot</SelectItem>
                  <SelectItem value="reorder">Profitable Reorders</SelectItem>
                </SelectContent>
              </Select>
            </div>
            {chartType !== 'scatter' && chartType !== 'reorder' && (
              <div className="flex flex-col gap-2">
                <label htmlFor="metric" className="text-sm font-medium">
                  Metric (Y-Axis)
                </label>
                <Select
                  value={metric}
                  onValueChange={(v) => setMetric(v as Metric)}
                >
                  <SelectTrigger id="metric">
                    <SelectValue placeholder="Select a metric" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Revenue">Revenue</SelectItem>
                    <SelectItem value="Profit">Profit</SelectItem>
                    <SelectItem value="Cost of Goods Sold">COGS</SelectItem>
                    <SelectItem value="Sales (Units)">Sales (Units)</SelectItem>
                     <SelectItem value="Opening Stock (Units)">
                      Opening Stock (Units)
                    </SelectItem>
                     <SelectItem value="Purchase (Units)">
                      Purchase (Units)
                    </SelectItem>
                    <SelectItem value="Closing Stock (Units)">
                      Closing Stock (Units)
                    </SelectItem>
                     <SelectItem value="Profit Margin (%)">Profit Margin (%)</SelectItem>
                     <SelectItem value="Stock Turnover Ratio">Stock Turnover Ratio</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            )}
             {chartType === 'scatter' && (
              <>
                <div className="flex flex-col gap-2">
                  <label htmlFor="scatter-x" className="text-sm font-medium">Metric (X-Axis)</label>
                  <Select value={scatterXMetric} onValueChange={(v) => setScatterXMetric(v as Metric)}>
                    <SelectTrigger id="scatter-x"><SelectValue /></SelectTrigger>
                    <SelectContent>
                       <SelectItem value="Sales (Units)">Sales (Units)</SelectItem>
                       <SelectItem value="Revenue">Revenue</SelectItem>
                       <SelectItem value="Profit">Profit</SelectItem>
                       <SelectItem value="Cost of Goods Sold">COGS</SelectItem>
                       <SelectItem value="Profit Margin (%)">Profit Margin (%)</SelectItem>
                       <SelectItem value="Stock Turnover Ratio">Stock Turnover Ratio</SelectItem>
                       <SelectItem value="Reorder Level">Reorder Level</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                 <div className="flex flex-col gap-2">
                  <label htmlFor="scatter-y" className="text-sm font-medium">Metric (Y-Axis)</label>
                  <Select value={scatterYMetric} onValueChange={(v) => setScatterYMetric(v as Metric)}>
                    <SelectTrigger id="scatter-y"><SelectValue /></SelectTrigger>
                    <SelectContent>
                       <SelectItem value="Profit">Profit</SelectItem>
                       <SelectItem value="Revenue">Revenue</SelectItem>
                       <SelectItem value="Sales (Units)">Sales (Units)</SelectItem>
                       <SelectItem value="Cost of Goods Sold">COGS</SelectItem>
                       <SelectItem value="Profit Margin (%)">Profit Margin (%)</SelectItem>
                       <SelectItem value="Stock Turnover Ratio">Stock Turnover Ratio</SelectItem>
                       <SelectItem value="Reorder Level">Reorder Level</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </>
            )}
            {chartType !== 'reorder' && (
              <div className="flex flex-col gap-2">
                <label htmlFor="dimension" className="text-sm font-medium">
                  Dimension (X-Axis / Grouping)
                </label>
                <Select
                  value={dimension}
                  onValueChange={(v) => setDimension(v as Dimension)}
                >
                  <SelectTrigger id="dimension">
                    <SelectValue placeholder="Select a dimension" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Item Name">Item Name</SelectItem>
                    <SelectItem value="SKU">SKU</SelectItem>
                    <SelectItem value="Category">Category</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            )}
          </div>
          <div className="w-full">{renderChart()}</div>
        </CardContent>
      </Card>
    </div>
  );
}
